<?php
/**
 * BAOBAB ZAMBEZE FARM - KAUNGE, TETE
 * Arquivo principal do sistema
 */

// Iniciar sessão
session_start();

// Definir constantes
define('ROOT_PATH', __DIR__);
define('APP_PATH', ROOT_PATH . '/app');
define('ASSETS_PATH', ROOT_PATH . '/assets');

// Carregar configurações
require_once ROOT_PATH . '/config.php';
require_once ROOT_PATH . '/Database.php';

// ============================================
// FUNÇÕES DO SISTEMA
// ============================================

// Funções de Sessão
function session_set($key, $value) {
    $_SESSION[$key] = $value;
}

function session_get($key, $default = null) {
    return isset($_SESSION[$key]) ? $_SESSION[$key] : $default;
}

function session_has($key) {
    return isset($_SESSION[$key]);
}

function session_delete($key) {
    if (isset($_SESSION[$key])) {
        unset($_SESSION[$key]);
    }
}

function session_flash($key, $value = null) {
    if ($value === null) {
        $val = session_get($key);
        session_delete($key);
        return $val;
    } else {
        session_set($key, $value);
    }
}

function is_logged_in() {
    return session_has('user_id');
}

function get_user_id() {
    return session_get('user_id');
}

function get_user_type() {
    return session_get('user_type');
}

function get_user_name() {
    return session_get('user_name');
}

// Funções de Formatação
function format_money($value) {
    return number_format($value, 2, ',', '.') . ' MZN';
}

function format_date($date) {
    return date('d/m/Y H:i', strtotime($date));
}

function format_date_only($date) {
    return date('d/m/Y', strtotime($date));
}

function format_time($time) {
    return date('H:i', strtotime($time));
}

// Funções de Segurança
function sanitize($data) {
    return htmlspecialchars(strip_tags(trim($data)), ENT_QUOTES, 'UTF-8');
}

function csrf_token() {
    if (!session_has('csrf_token')) {
        session_set('csrf_token', bin2hex(random_bytes(32)));
    }
    return session_get('csrf_token');
}

function verify_csrf() {
    $token = $_POST['csrf_token'] ?? '';
    return hash_equals(session_get('csrf_token') ?? '', $token);
}

// Funções de Navegação
function redirect($url) {
    header("Location: " . SITE_URL . $url);
    exit();
}

function url($path = '') {
    return SITE_URL . $path;
}

function asset($path) {
    return ASSETS_URL . '/' . ltrim($path, '/');
}

// Funções de Request
function is_post() {
    return $_SERVER['REQUEST_METHOD'] === 'POST';
}

function is_get() {
    return $_SERVER['REQUEST_METHOD'] === 'GET';
}

function post($key, $default = '') {
    return isset($_POST[$key]) ? sanitize($_POST[$key]) : $default;
}

function get_param($key, $default = '') {
    return isset($_GET[$key]) ? sanitize($_GET[$key]) : $default;
}

function old($key, $default = '') {
    return isset($_POST[$key]) ? sanitize($_POST[$key]) : $default;
}

// Funções de Validação
function validate_required($value) {
    return !empty(trim($value));
}

function validate_email($email) {
    return filter_var($email, FILTER_VALIDATE_EMAIL);
}

function validate_min_length($value, $min) {
    return strlen($value) >= $min;
}

function validate_max_length($value, $max) {
    return strlen($value) <= $max;
}

// Funções de Debug
function dd($data) {
    echo '<pre>';
    var_dump($data);
    echo '</pre>';
    die();
}

function dump($data) {
    echo '<pre>';
    print_r($data);
    echo '</pre>';
}

// Funções de View
function view($view, $data = []) {
    extract($data);
    $viewPath = APP_PATH . '/views/' . $view . '.php';
    
    if (file_exists($viewPath)) {
        require_once $viewPath;
    } else {
        die("View não encontrada: " . $view);
    }
}

function json_response($data, $status = 200) {
    http_response_code($status);
    header('Content-Type: application/json');
    echo json_encode($data);
    exit();
}

// ============================================
// AUTOLOAD DE CLASSES
// ============================================

spl_autoload_register(function($class) {
    $paths = [
        APP_PATH . '/controllers/' . $class . '.php',
        APP_PATH . '/models/' . $class . '.php'
    ];
    
    foreach ($paths as $path) {
        if (file_exists($path)) {
            require_once $path;
            return;
        }
    }
});

// ============================================
// SISTEMA DE ROTAS COM ERROR 404/403
// ============================================

$url = isset($_GET['url']) ? explode('/', filter_var(rtrim($_GET['url'], '/'), FILTER_SANITIZE_URL)) : [];

// Controller e método padrão
$controllerName = 'IndexController';
$method = 'index';
$params = [];

// Verificar se usuário está logado
$isLoggedIn = is_logged_in();

// Processar URL
if (isset($url[0]) && !empty($url[0])) {
    $controllerName = ucfirst($url[0]) . 'Controller';
    unset($url[0]);
}

// Verificar se método existe
if (isset($url[1]) && !empty($url[1])) {
    $method = $url[1];
    unset($url[1]);
}

// Parâmetros restantes
$params = $url ? array_values($url) : [];

// Verificar se controller existe
$controllerFile = APP_PATH . '/controllers/' . $controllerName . '.php';
if (!file_exists($controllerFile)) {
    // Controller não existe
    if ($isLoggedIn) {
        // Usuário logado: mostrar 404
        $controller = new ErrorController();
        $controller->error404();
        exit();
    } else {
        // Usuário não logado: mostrar 403
        $controller = new ErrorController();
        $controller->error403();
        exit();
    }
}

// Controllers públicos (acessíveis sem login)
$publicControllers = [
    'IndexController',
    'HomeController',           // ATUALIZADO: AboutController -> HomeController
    'ProductionController',
    'ActivitiesController',
    'ContactController',
    'ErrorController'
];

// Verificar se precisa estar logado
if (!$isLoggedIn && !in_array($controllerName, $publicControllers)) {
    // Usuário não logado tentando acessar área restrita: mostrar 403
    $controller = new ErrorController();
    $controller->error403();
    exit();
}

// Instanciar controller
$controller = new $controllerName;

// Verificar se método existe
if (method_exists($controller, $method)) {
    call_user_func_array([$controller, $method], $params);
} else {
    // Método não existe
    if ($isLoggedIn) {
        // Usuário logado: mostrar 404
        $errorController = new ErrorController();
        $errorController->error404();
        exit();
    } else {
        // Usuário não logado: mostrar 403
        $errorController = new ErrorController();
        $errorController->error403();
        exit();
    }
}

function is_active_url($path) {
    $currentUrl = isset($_GET['url']) ? $_GET['url'] : '';
    $currentPath = explode('/', $currentUrl)[0];
    $checkPath = ltrim($path, '/');
    
    return $currentPath === $checkPath ? 'active' : '';
}