// Hero Carousel
let currentSlide = 0;
const slides = document.querySelectorAll('.carousel-slide');
const totalSlides = slides.length;

// Extrair textos dos data-attributes dos slides
const heroTexts = Array.from(slides).map(slide => ({
    title: slide.getAttribute('data-title') || '',
    subtitle: slide.getAttribute('data-subtitle') || ''
}));

// Extrair dados dos produtos dinamicamente do DOM
const productData = [];
document.querySelectorAll('.product-card').forEach((card, cardIndex) => {
    const images = card.querySelectorAll('.product-image img');
    const cardProducts = [];
    
    images.forEach(img => {
        cardProducts.push({
            title: img.alt,
            slug: img.getAttribute('data-product')
        });
    });
    
    productData.push(cardProducts);
});

// Pegar TODAS as estatísticas do banco (injetadas pelo PHP)
const allStatistics = window.allStatisticsData || [];

console.log('=== INICIALIZAÇÃO ===');
console.log('Total de estatísticas:', allStatistics.length);
console.log('Dados:', allStatistics);

// Controle da rotação de estatísticas
let statRotationIndex = 0; // Índice global da próxima estatística a mostrar
let statCardToUpdate = 0;  // Qual card (0, 1 ou 2) vai ser atualizado

let currentProductRotation = [0, 0, 0];

function updateSlide() {
    slides.forEach((slide, index) => {
        slide.classList.toggle('active', index === currentSlide);
    });

    const heroTitle = document.getElementById('heroTitle');
    const heroSubtitle = document.getElementById('heroSubtitle');

    if (heroTitle && heroSubtitle && heroTexts[currentSlide]) {
        heroTitle.style.opacity = '0';
        heroSubtitle.style.opacity = '0';

        setTimeout(() => {
            heroTitle.innerHTML = heroTexts[currentSlide].title;
            heroSubtitle.innerHTML = heroTexts[currentSlide].subtitle;
            heroTitle.style.opacity = '1';
            heroSubtitle.style.opacity = '1';
        }, 300);
    }
}

function nextSlide() {
    currentSlide = (currentSlide + 1) % totalSlides;
    updateSlide();
}

function prevSlide() {
    currentSlide = (currentSlide - 1 + totalSlides) % totalSlides;
    updateSlide();
}

// Product cards rotation
function rotateProductCard(cardIndex) {
    const card = document.querySelectorAll('.product-card')[cardIndex];
    if (!card) return;
    
    const images = card.querySelectorAll('.product-image img');
    const title = document.getElementById(`productTitle${cardIndex + 1}`);
    const detailTexts = card.querySelectorAll('.detail-text');

    if (images.length === 0 || !title || detailTexts.length === 0) return;

    images[currentProductRotation[cardIndex]].classList.remove('active');
    detailTexts[currentProductRotation[cardIndex]].classList.remove('active');

    currentProductRotation[cardIndex] = (currentProductRotation[cardIndex] + 1) % images.length;

    images[currentProductRotation[cardIndex]].classList.add('active');
    detailTexts[currentProductRotation[cardIndex]].classList.add('active');
    
    if (productData[cardIndex] && productData[cardIndex][currentProductRotation[cardIndex]]) {
        title.innerHTML = productData[cardIndex][currentProductRotation[cardIndex]].title;
    }
}

// Formatação de números
function formatNumber(num) {
    return num.toString().replace(/\B(?=(\d{3})+(?!\d))/g, ".");
}

// Animação de números
function animateNumber(element, targetValue, duration = 2000) {
    const startValue = 0;
    const startTime = performance.now();

    function updateNumber(currentTime) {
        const elapsed = currentTime - startTime;
        const progress = Math.min(elapsed / duration, 1);
        const easeOutQuad = 1 - Math.pow(1 - progress, 2);
        const currentValue = Math.floor(startValue + (targetValue - startValue) * easeOutQuad);
        const formattedValue = formatNumber(currentValue);
        element.textContent = `${formattedValue} Kg`;

        if (progress < 1) {
            requestAnimationFrame(updateNumber);
        } else {
            element.textContent = `${formatNumber(targetValue)} Kg`;
        }
    }

    requestAnimationFrame(updateNumber);
}

// NOVA LÓGICA DE ROTAÇÃO DE ESTATÍSTICAS - SIMPLES E DIRETA
function rotateStatistics() {
    if (allStatistics.length <= 3) {
        console.log('Apenas 3 ou menos estatísticas, rotação desativada');
        return;
    }
    
    console.log(`\n=== ROTAÇÃO #${statRotationIndex} ===`);
    console.log(`Atualizando Card ${statCardToUpdate + 1}`);
    console.log(`Próxima estatística: índice ${statRotationIndex}`);
    
    const statImg = document.getElementById(`statImg${statCardToUpdate + 1}`);
    const statValue = document.getElementById(`statValue${statCardToUpdate + 1}`);
    const statLabel = document.getElementById(`statLabel${statCardToUpdate + 1}`);

    if (!statImg || !statValue || !statLabel) {
        console.error('Elementos não encontrados!');
        return;
    }

    const data = allStatistics[statRotationIndex];
    console.log('Dados:', data);

    // Fade out
    statValue.style.opacity = '0';
    statLabel.style.opacity = '0';
    statImg.style.opacity = '0';

    setTimeout(() => {
        // Atualizar conteúdo
        statImg.src = data.image;
        statImg.alt = data.label;
        statLabel.textContent = data.label;

        // Fade in
        statImg.style.opacity = '1';
        statLabel.style.opacity = '1';
        animateNumber(statValue, data.value, 1500);
        statValue.style.opacity = '1';
        
        console.log(`✅ Card ${statCardToUpdate + 1} atualizado: ${data.label}`);
    }, 300);
    
    // Avançar para próxima estatística (circular)
    statRotationIndex = (statRotationIndex + 1) % allStatistics.length;
    
    // Avançar para próximo card (0 → 1 → 2 → 0)
    statCardToUpdate = (statCardToUpdate + 1) % 3;
}

// Inicializar estatísticas (mostrar as 3 primeiras)
function initializeStatistics() {
    console.log('=== INICIALIZANDO ESTATÍSTICAS ===');
    for (let i = 0; i < 3 && i < allStatistics.length; i++) {
        const statValue = document.getElementById(`statValue${i + 1}`);
        if (statValue && allStatistics[i]) {
            setTimeout(() => {
                animateNumber(statValue, allStatistics[i].value);
                console.log(`Card ${i + 1} inicializado: ${allStatistics[i].label}`);
            }, i * 200);
        }
    }
    
    // Começar rotação a partir do índice 3
    statRotationIndex = 3;
    statCardToUpdate = 0;
}

// Auto-advance carousel
let autoSlideInterval = setInterval(nextSlide, 3000);

// PRODUTOS - Rotação
let productRotationTimer;
function startProductRotation() {
    let cardSequence = 0;
    productRotationTimer = setTimeout(function rotate() {
        rotateProductCard(cardSequence);
        cardSequence = (cardSequence + 1) % 3;
        productRotationTimer = setTimeout(rotate, 5000);
    }, 5000);
}

if (document.querySelectorAll('.product-card').length > 0) {
    startProductRotation();
}

// ESTATÍSTICAS - Rotação a cada 5 segundos
let statisticsRotationTimer;
function startStatisticsRotation() {
    if (allStatistics.length <= 3) {
        console.log('Rotação de estatísticas desativada (3 ou menos itens)');
        return;
    }
    
    console.log('🔄 Iniciando rotação de estatísticas...');
    statisticsRotationTimer = setInterval(() => {
        rotateStatistics();
    }, 5000);
}

// Inicializar após 1 segundo
setTimeout(() => {
    initializeStatistics();
    
    // Iniciar rotação apenas se houver mais de 3 estatísticas
    if (allStatistics.length > 3) {
        startStatisticsRotation();
    }
}, 1000);

// Pause carousel on hover
const carousel = document.getElementById('heroCarousel');
if (carousel) {
    carousel.addEventListener('mouseenter', () => {
        clearInterval(autoSlideInterval);
    });

    carousel.addEventListener('mouseleave', () => {
        autoSlideInterval = setInterval(nextSlide, 3000);
    });
}

// Mobile menu toggle
function toggleMobileMenu() {
    const navMenu = document.getElementById('navMenu');
    if (navMenu) {
        navMenu.classList.toggle('active');
    }
}