-- =============================================
-- TABELA: layout_head
-- Armazena metadados e informações do <head>
-- =============================================
CREATE TABLE layout_head (
    id INT AUTO_INCREMENT PRIMARY KEY,
    page_identifier VARCHAR(50) UNIQUE NOT NULL COMMENT 'Identificador da página (ex: home, production, activities, contact)',
    page_title VARCHAR(255) NOT NULL,
    meta_description TEXT,
    meta_keywords TEXT,
    meta_author VARCHAR(100),
    og_title VARCHAR(255),
    og_description TEXT,
    og_image VARCHAR(255),
    og_url VARCHAR(255),
    twitter_title VARCHAR(255),
    twitter_description TEXT,
    twitter_image VARCHAR(255),
    canonical_url VARCHAR(255),
    favicon_url VARCHAR(255),
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- =============================================
-- TABELA: layout_header
-- Armazena informações do menu de navegação
-- =============================================
CREATE TABLE layout_header (
    id INT AUTO_INCREMENT PRIMARY KEY,
    logo_url VARCHAR(255) NOT NULL,
    logo_alt VARCHAR(100),
    logo_width INT DEFAULT 50,
    logo_height INT DEFAULT 50,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- Tabela de itens do menu
CREATE TABLE header_menu_items (
    id INT AUTO_INCREMENT PRIMARY KEY,
    header_id INT NOT NULL,
    label VARCHAR(100) NOT NULL,
    url VARCHAR(255) NOT NULL,
    display_order INT DEFAULT 0,
    is_active BOOLEAN DEFAULT TRUE,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    FOREIGN KEY (header_id) REFERENCES layout_header(id) ON DELETE CASCADE
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- =============================================
-- TABELA: page_home (Página Principal - About)
-- =============================================
CREATE TABLE page_home (
    id INT AUTO_INCREMENT PRIMARY KEY,
    page_identifier VARCHAR(50) UNIQUE NOT NULL,
    
    -- Hero Section
    hero_title VARCHAR(255),
    hero_subtitle TEXT,
    
    -- Cards de Missão, Visão e Valores
    mission_title VARCHAR(100),
    mission_content TEXT,
    
    vision_title VARCHAR(100),
    vision_content TEXT,
    
    values_title VARCHAR(100),
    values_content JSON COMMENT 'Array de valores em formato JSON',
    
    -- História
    history_title VARCHAR(255),
    history_content TEXT,
    
    is_active BOOLEAN DEFAULT TRUE,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- Tabela para imagens do carousel da home
CREATE TABLE page_home_carousel (
    id INT AUTO_INCREMENT PRIMARY KEY,
    page_home_id INT NOT NULL,
    image_url VARCHAR(255) NOT NULL,
    image_alt VARCHAR(255),
    display_order INT DEFAULT 0,
    is_active BOOLEAN DEFAULT TRUE,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    FOREIGN KEY (page_home_id) REFERENCES page_home(id) ON DELETE CASCADE
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- =============================================
-- TABELA: page_production (Página de Produção)
-- =============================================
CREATE TABLE page_production (
    id INT AUTO_INCREMENT PRIMARY KEY,
    page_identifier VARCHAR(50) UNIQUE NOT NULL,
    
    -- Seção de Estatísticas
    statistics_title VARCHAR(255),
    statistics_subtitle TEXT,
    
    is_active BOOLEAN DEFAULT TRUE,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- Tabela para carousel de produção
CREATE TABLE production_carousel (
    id INT AUTO_INCREMENT PRIMARY KEY,
    production_id INT NOT NULL,
    image_url VARCHAR(255) NOT NULL,
    image_alt VARCHAR(255),
    title VARCHAR(255),
    subtitle TEXT,
    display_order INT DEFAULT 0,
    is_active BOOLEAN DEFAULT TRUE,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    FOREIGN KEY (production_id) REFERENCES page_production(id) ON DELETE CASCADE
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- Tabela para produtos
CREATE TABLE production_products (
    id INT AUTO_INCREMENT PRIMARY KEY,
    production_id INT NOT NULL,
    card_number INT NOT NULL COMMENT 'Número do card (1, 2, 3)',
    product_name VARCHAR(100) NOT NULL,
    product_slug VARCHAR(100) NOT NULL,
    image_url VARCHAR(255) NOT NULL,
    description TEXT,
    display_order INT DEFAULT 0,
    is_main BOOLEAN DEFAULT FALSE COMMENT 'Se é o produto principal do card',
    is_active BOOLEAN DEFAULT TRUE,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    FOREIGN KEY (production_id) REFERENCES page_production(id) ON DELETE CASCADE
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- Tabela para estatísticas de produção
CREATE TABLE production_statistics (
    id INT AUTO_INCREMENT PRIMARY KEY,
    production_id INT NOT NULL,
    product_name VARCHAR(100) NOT NULL,
    image_url VARCHAR(255) NOT NULL,
    value VARCHAR(50) NOT NULL COMMENT 'Valor da produção (ex: 25t, 15t)',
    display_order INT DEFAULT 0,
    is_active BOOLEAN DEFAULT TRUE,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    FOREIGN KEY (production_id) REFERENCES page_production(id) ON DELETE CASCADE
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- =============================================
-- TABELA: page_activities (Página de Actividades)
-- =============================================
CREATE TABLE page_activities (
    id INT AUTO_INCREMENT PRIMARY KEY,
    page_identifier VARCHAR(50) UNIQUE NOT NULL,
    
    -- Hero Section
    hero_initial_title VARCHAR(255),
    hero_initial_subtitle TEXT,
    
    -- Seções
    social_practices_title VARCHAR(255),
    agricultural_practices_title VARCHAR(255),
    
    is_active BOOLEAN DEFAULT TRUE,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- Tabela para slides de actividades
CREATE TABLE activities_slides (
    id INT AUTO_INCREMENT PRIMARY KEY,
    activities_id INT NOT NULL,
    slide_number INT NOT NULL,
    image_url VARCHAR(255) NOT NULL,
    image_alt VARCHAR(255),
    title VARCHAR(255),
    description TEXT,
    category VARCHAR(50) COMMENT 'social ou agricultural',
    display_order INT DEFAULT 0,
    is_active BOOLEAN DEFAULT TRUE,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    FOREIGN KEY (activities_id) REFERENCES page_activities(id) ON DELETE CASCADE
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- Tabela para cards de actividades
CREATE TABLE activities_cards (
    id INT AUTO_INCREMENT PRIMARY KEY,
    activities_id INT NOT NULL,
    card_title VARCHAR(255) NOT NULL,
    image_url VARCHAR(255) NOT NULL,
    image_alt VARCHAR(255),
    category VARCHAR(50) NOT NULL COMMENT 'social ou agricultural',
    linked_slide_number INT COMMENT 'Número do slide relacionado',
    display_order INT DEFAULT 0,
    is_active BOOLEAN DEFAULT TRUE,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    FOREIGN KEY (activities_id) REFERENCES page_activities(id) ON DELETE CASCADE
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- =============================================
-- TABELA: page_contact (Página de Contacto)
-- =============================================
CREATE TABLE page_contact (
    id INT AUTO_INCREMENT PRIMARY KEY,
    page_identifier VARCHAR(50) UNIQUE NOT NULL,
    
    -- Hero Section
    hero_title VARCHAR(255),
    
    -- Form Section
    form_title VARCHAR(255),
    
    -- Info Section
    location_title VARCHAR(100),
    location_content TEXT,
    
    email_title VARCHAR(100),
    email_content VARCHAR(255),
    
    -- Gallery
    gallery_title VARCHAR(255),
    
    is_active BOOLEAN DEFAULT TRUE,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- Tabela para galeria de imagens do contacto
CREATE TABLE contact_gallery (
    id INT AUTO_INCREMENT PRIMARY KEY,
    contact_id INT NOT NULL,
    image_url VARCHAR(255) NOT NULL,
    image_alt VARCHAR(255),
    display_order INT DEFAULT 0,
    is_active BOOLEAN DEFAULT TRUE,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    FOREIGN KEY (contact_id) REFERENCES page_contact(id) ON DELETE CASCADE
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- Tabela para mensagens de contacto
CREATE TABLE contact_messages (
    id INT AUTO_INCREMENT PRIMARY KEY,
    first_name VARCHAR(100) NOT NULL,
    last_name VARCHAR(100) NOT NULL,
    email VARCHAR(255) NOT NULL,
    message TEXT NOT NULL,
    is_read BOOLEAN DEFAULT FALSE,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- =============================================
-- TABELA: page_footer
-- =============================================
CREATE TABLE page_footer (
    id INT AUTO_INCREMENT PRIMARY KEY,
    
    policies_title VARCHAR(100),
    social_title VARCHAR(100),
    contacts_title VARCHAR(100),
    
    company_name VARCHAR(255),
    address_line1 VARCHAR(255),
    address_line2 VARCHAR(255),
    address_line3 VARCHAR(255),
    address_line4 VARCHAR(255),
    phone1 VARCHAR(20),
    phone2 VARCHAR(20),
    email_primary VARCHAR(100),
    email_secondary VARCHAR(100),
    
    copyright_text TEXT,
    
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

CREATE TABLE footer_policy_links (
    id INT AUTO_INCREMENT PRIMARY KEY,
    footer_id INT NOT NULL,
    label VARCHAR(100) NOT NULL,
    url VARCHAR(255) NOT NULL,
    display_order INT DEFAULT 0,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    FOREIGN KEY (footer_id) REFERENCES page_footer(id) ON DELETE CASCADE
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

CREATE TABLE footer_social_links (
    id INT AUTO_INCREMENT PRIMARY KEY,
    footer_id INT NOT NULL,
    platform VARCHAR(50) NOT NULL,
    icon_class VARCHAR(100) NOT NULL,
    url VARCHAR(255) NOT NULL,
    display_order INT DEFAULT 0,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    FOREIGN KEY (footer_id) REFERENCES page_footer(id) ON DELETE CASCADE
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- =============================================
-- INSERÇÃO DE DADOS
-- =============================================

-- Inserir dados do HEADER
INSERT INTO layout_header (logo_url, logo_alt, logo_width, logo_height)
VALUES ('/assets/images/logo.png', 'Baobab Zambeze Farm', 50, 50);

SET @header_id = LAST_INSERT_ID();

INSERT INTO header_menu_items (header_id, label, url, display_order)
VALUES 
    (@header_id, 'Sobre Nós', 'about.html', 1),
    (@header_id, 'Produção', 'production.html', 2),
    (@header_id, 'Actividades', 'activities.html', 3),
    (@header_id, 'Contacto', 'contact.html', 4);

-- Inserir dados do FOOTER
INSERT INTO page_footer (policies_title, social_title, contacts_title, 
    company_name, address_line1, address_line2, address_line3,
    phone1, phone2, email_primary, email_secondary, copyright_text)
VALUES (
    'Políticas',
    'Redes Sociais',
    'Contactos',
    'Baobab Zambeze Farm',
    'Povoado de Kaunge, Distrito de Moatize',
    'Província de Tete, Moçambique',
    'EN9 - A caminho da Zâmbia',
    '862812986',
    '846632923',
    'info@baobabzambeze.farm',
    'paulo.borges@baobabzambeze.farm',
    '© 2024 Baobab Zambeze Farm. Todos os direitos reservados. Desenvolvido por Labore'
);

SET @footer_id = LAST_INSERT_ID();

INSERT INTO footer_policy_links (footer_id, label, url, display_order)
VALUES 
    (@footer_id, 'Políticas de Privacidade', '/privacy.html', 1),
    (@footer_id, 'Termos e Condições', '/terms.html', 2);

INSERT INTO footer_social_links (footer_id, platform, icon_class, url, display_order)
VALUES 
    (@footer_id, 'facebook', 'fab fa-facebook-f', '#', 1),
    (@footer_id, 'instagram', 'fab fa-instagram', '#', 2);